<?php
/**
 * This file handles helper function.
 *
 * @category Helpers
 * @package  OxyProps
 * @author   Cédric Bontems <cedric@thewebforge.dev>
 * @license  https://www.gnu.org/licenses/gpl-2.0.html  GPL v2 or later
 * @link     https://oxyprops.com                       OxyProps Website
 * @since    1.4.0
 */

namespace OxyProps\Inc;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Helpers methods
 *
 * @since 1.0.0
 */
class Helpers extends Base_Controller {


	/**
	 * Unique instance of Helpers
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @var object Self stored instance
	 */
	private static $instance = null;

	/**
	 * Returns the unique instance of Helpers
	 * and creates one if it does not already exist.
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return object Dashboard singleton.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new Helpers();
		}
		return self::$instance;
	}

	/**
	 * Get SVG icon markup (<use>)
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @param  string  $name Icon name.
	 * @param  boolean $link Return a link url or not.
	 * @param  boolean $wants_to_echo Echo or Return markup.
	 * @return string
	 */
	public function get_icon( $name = null, $link = false, $wants_to_echo = true ) {
		global $is_IE;
		if ( ! $name ) {
			return;
		}
		$icon = 'oxyprops-' . $name . '-icon';
		if ( $is_IE ) {
			$url = '#' . $icon;
		} else {
			$url = $this->oxyprops_url . 'public/svg/sprite.svg?v=' . $this->oxyprops_version . '#' . $icon;
		}
		if ( $link ) {
			return esc_url( $url );
		}
		if ( ! $wants_to_echo ) {
			ob_start();
		}
		echo '<svg class="' . sanitize_html_class( $icon ) . '">';
		echo '<use xlink:href="' . esc_url( $url ) . '"></use>';
		echo '</svg>';
		if ( ! $wants_to_echo ) {
			return ob_get_clean();
		}
	}

	/**
	 * Get user capability
	 * Defaults to manage_options
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return string
	 */
	public static function get_user_capability() {
		return apply_filters( 'oxyprops_user_capability', 'manage_options' );
	}

	/**
	 * Check user capability
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return boolean
	 */
	public static function current_user_can() {
		$capability = self::get_user_capability();
		return current_user_can( $capability );
	}

	/**
	 * Replace Oxygen Builder Global Settings by OxyProps
	 * default values
	 *
	 * @return void
	 * @since 0.2.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public function oxy_settings_override() {
		// Record the current state in database.
		$back_up = get_option( 'ct_global_settings' );
		$this->update_plugin_current_settings( 'oxy_backup', $back_up );

		// Replace Oxygen global settings with OxyProps Defaults.
		$defaults                  = Defaults::get_instance();
		$oxyprops_default_settings = $defaults->get_oxyprops_global_settings_for_oxygen();
		update_option( 'ct_global_settings', $oxyprops_default_settings );
	}

	/**
	 * Add OxyProps theme to Bricks
	 * default values
	 *
	 * @return bool
	 * @since 0.2.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public function bricks_settings_override() {
		// Record the current state in database.
		$defaults                = Defaults::get_instance();
		$back_up                 = get_option( 'bricks_theme_styles' );
		$current_oxyprops_theme  = $back_up['oxyprops'];
		$new_bricks_theme_option = $back_up;
		$this->update_plugin_current_settings( 'bricks_backup', $back_up );
		$new_bricks_theme_option['oxyprops'] = $defaults->get_oxyprops_global_settings_for_bricks();

		// If no theme in the DB, create it.
		if ( ! $current_oxyprops_theme ) {
			return update_option( 'bricks_theme_styles', $new_bricks_theme_option );
		}
		return update_option( 'bricks_theme_styles', array_replace_recursive( $new_bricks_theme_option, $back_up ) );
	}

	/**
	 * Search strings with wildcards
	 *
	 * @since 1.7.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 * @param string $wildcard_pattern example: 'oxyprops_*_icon'.
	 * @param string $haystack the string to be searched.
	 * @return int|false
	 */
	public static function match_wildcard( $wildcard_pattern, $haystack ) {
		$regex = str_replace(
			array( '\*', '\?' ),
			array( '.*', '.' ),
			preg_quote( $wildcard_pattern, '/' )
		);
		return preg_match( '/^' . $regex . '$/is', $haystack );
	}

	/**
	 * Patch Bricks
	 *
	 * @return void
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public function brx_mute() {
		self::brx_mute_assets();
		self::brx_mute_iframe_min_js();
		self::brx_mute_main_min_js();
		self::brx_mute_frontend_min_css();
	}

	/**
	 * Patch Bricks Assets
	 *
	 * @return void
	 * @since 1.4.0
	 * @since 1.11.3 Switch to WP_Filesystem.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public static function brx_mute_assets() {
		global $wp_filesystem;
		include_once ABSPATH . 'wp-admin/includes/file.php';
		WP_Filesystem();

		$target_file = get_template_directory() . '/includes/assets.php';
		if ( ! $wp_filesystem->exists( $target_file ) ) {
			return;
		}

		// read the entire string.
		$str = $wp_filesystem->get_contents( $target_file );
		// Check for existing patch.
		if ( null === $str || str_contains( $str, 'OxyProps Patch - Start' ) ) {
			return;
		}

		$initial_length = strlen( $str );
		$search         = '$css_rules[ $css_selector ][] = "$font_property: \"$font_value\"$fallback_font";';
		// Bricks 1.5.7 -> Line 1534.
		// Bricks 1.6 Beta 2 -> Line 1587.
		// Bricks 1.7 Beta -> Line 1634.
		// Many tabs at this position in source file.
		$tab     = chr( 9 ) . chr( 9 ) . chr( 9 ) . chr( 9 ) . chr( 9 ) . chr( 9 ) . chr( 9 ) . chr( 9 ) . chr( 9 ) . chr( 9 );
		$replace = '// OxyProps Patch - Start.' . PHP_EOL . $tab .
			'// Well, always add quotes **unless** you are using CSS Variables.' . PHP_EOL . $tab .
			'$css_rules[ $css_selector ][] = str_contains( $font_value, \'var(--o-\' ) ? "$font_property: $font_value $fallback_font" : "$font_property: \"$font_value\"$fallback_font";' . PHP_EOL . $tab .
			'// Just in case, here is the original line of code :' . PHP_EOL . $tab .
			'// $css_rules[ $css_selector ][] = "$font_property: \"$font_value\"$fallback_font";' . PHP_EOL . $tab .
			'// OxyProps Patch - End.' . PHP_EOL;
		$str     = str_replace( $search, $replace, $str );
		// write changes if patch was actually applied.
		if ( strlen( $str ?? '' ) > $initial_length ) {
			$wp_filesystem->put_contents( $target_file, $str );
		}
	}

	/**
	 * Patch Bricks JS
	 *
	 * @since 1.7.0  Creation.
	 * @since 1.11.3 Fix for Bricks 1.9.5.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 * @param string $str the string to be patched.
	 * @return string | null
	 */
	public static function brx_mute_js( $str ) {
		$pattern = '/if\(\w+\.push\(\'font-family: "\'\.concat\(\w+,\'"\'\)\.concat\(\w+\)\)/m';

		if ( 1 !== preg_match( $pattern, $str, $matches ) ) {
			return null;
		}
		if ( 1 !== preg_match( '/\w+\./', $matches[0], $f1_matches ) || 1 !== preg_match( '/\w+,/', $matches[0], $f2_matches ) || 1 !== preg_match( '/\w+\)/', $matches[0], $f3_matches ) ) {
			return null;
		}

		$f1 = substr( $f1_matches[0], 0, -1 );
		$f2 = substr( $f2_matches[0], 0, -1 );
		$f3 = substr( $f3_matches[0], 0, -1 );

		$replacement = '/* OxyProps Patch - Start */if(' . $f1 . '.push(typeof ' . $f2 . ' === "string" && ' . $f2 . '.substring(0, 8) === "var(--o-" ? "font-family: ".concat(' . $f2 . ').concat(' . $f3 . ') : \'font-family: "\'.concat(' . $f2 . ', \'"\').concat(' . $f3 . '))/* OxyProps Patch - End */';
		return preg_replace(
			$pattern,
			$replacement,
			$str
		);
	}

	/**
	 * Patch Bricks JS file
	 *
	 * @since 1.7.0  Fix for Bricks 1.8
	 * @since 1.11.3 Switch to WP_Filesystem.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 * @param string $file the file to be patched.
	 * @return void
	 */
	public static function brx_mute_js_file( $file ) {
		global $wp_filesystem;
		require_once ABSPATH . '/wp-admin/includes/file.php';
		WP_Filesystem();

		$target_file = get_template_directory() . $file;
		if ( ! $wp_filesystem->exists( $target_file ) ) {
			return;
		}
		// read the entire string.
		$str = $wp_filesystem->get_contents( $target_file );

		if ( null === $str ) {
			return;
		}

		$result = self::brx_mute_js( $str );

		if ( strlen( $result ?? '' ) > strlen( $str ) ) {
			$wp_filesystem->put_contents( $target_file, $result, FS_CHMOD_FILE );
		}
	}

	/**
	 * Patch Bricks Iframe JS
	 *
	 * @return void
	 * @since 1.4.0
	 * @since 1.4.13 Fix for Bricks 1.7 Beta
	 * @since 1.7.0  Fix for Bricks 1.8
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public static function brx_mute_iframe_min_js() {
		$target_file = '/assets/js/iframe.min.js';
		self::brx_mute_js_file( $target_file );
	}

	/**
	 * Patch Bricks Main JS
	 *
	 * @return void
	 * @since 1.4.0
	 * @since 1.4.13 Fix for Bricks 1.7 Beta
	 * @since 1.7.0  Fix for Bricks 1.8
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public static function brx_mute_main_min_js() {
		$target_file = '/assets/js/main.min.js';
		self::brx_mute_js_file( $target_file );
	}

	/**
	 * Patch Bricks Frontend CSS
	 *
	 * @return void
	 * @since 1.5.0 Creation
	 * @since 1.11.3 Switch to WP_Filesystem.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public static function brx_mute_frontend_min_css() {
		global $wp_filesystem;
		require_once ABSPATH . '/wp-admin/includes/file.php';
		WP_Filesystem();

		$target_file = get_template_directory() . '/assets/css/frontend.min.css';
		if ( ! $wp_filesystem->is_writable( $target_file ) ) {
			return;
		}
		// read the entire string.
		$str = $wp_filesystem->get_contents( $target_file );
		if ( null === $str || str_contains( $str, 'OxyProps Patch - Start' ) ) {
			return;
		}
		$initial_length = strlen( $str );
		$search         = 'font-family:georgia,Times New Roman,times,serif;';
		$replace        = '/* OxyProps Patch - Start */font-family:inherit;/* OxyProps Patch - End */';
		$str            = str_replace( $search, $replace, $str );

		// write the entire string if patch was actually applied.
		if ( strlen( $str ?? '' ) > $initial_length ) {
			$wp_filesystem->put_contents( $target_file, $str );
		}
	}

	/**
	 * Patch Oxygen
	 *
	 * @return void
	 * @since 0.3.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public function oxy_mute() {
		self::oxy_mute_component_init();
		self::oxy_mute_controller_js();
		self::oxy_mute_iframe_js();
	}
	/**
	 * Patch Oxygen Component Init
	 *
	 * @return void
	 * @since 0.3.0
	 * @since 1.11.3 Switch to WP_Filesystem.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public static function oxy_mute_component_init() {
		global $wp_filesystem;
		include_once ABSPATH . 'wp-admin/includes/file.php';
		WP_Filesystem();

		$target_file = WP_PLUGIN_DIR . '/oxygen/component-framework/component-init.php';
		if ( ! \wp_is_writable( $target_file ) ) {
			return;
		}

		// read the entire string.
		$str = $wp_filesystem->get_contents( $target_file );
		if ( str_contains( $str, 'var(--o-' ) ) {
			return;
		}

		$search  = '$text_font = "\'$text_font\'";';
		$replace = '$text_font = (substr("$text_font", 0, 8) == \'var(--o-\') ? "$text_font" : "\'$text_font\'";';
		$str     = str_replace( $search, $replace, $str );

		$search  = '$display_font = "\'$display_font\'";';
		$replace = '$display_font = (substr("$display_font", 0, 8) == \'var(--o-\') ? "$display_font" : "\'$display_font\'";';
		$str     = str_replace( $search, $replace, $str );

		$search  = 'array_push($styles, \'url(\'.do_shortcode($stateOptions[\'background-image\']).\')\');';
		$replace = 'array_push($styles, (str_contains($stateOptions[\'background-image\'], \'var(--o-\')  || str_contains( $stateOptions[\'background-image\'], \'gradient\') ? \'\' : \'url(\') . do_shortcode($stateOptions[\'background-image\']) . (str_contains($stateOptions[\'background-image\'], \'var(--o-\') || str_contains( $stateOptions[\'background-image\'], \'gradient\') ? \'\' : \')\'));';
		$str     = str_replace( $search, $replace, $str );

		// write the entire string.
		$wp_filesystem->put_contents( $target_file, $str );
	}

	/**
	 * Patch Oxygen CSS Controller
	 *
	 * @return void
	 * @since 0.3.0
	 * @since 1.11.3 Switch to WP_Filesystem.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public static function oxy_mute_controller_js() {
		global $wp_filesystem;
		require_once ABSPATH . '/wp-admin/includes/file.php';
		WP_Filesystem();

		$target_file = WP_PLUGIN_DIR . '/oxygen/component-framework/angular/controllers/controller.css.js';
		if ( ! $wp_filesystem->is_writable( $target_file ) ) {
			return;
		}

		// read the entire string.
		$str = $wp_filesystem->get_contents( $target_file );
		if ( str_contains( $str, 'var(--o-' ) ) {
			return;
		}

		$search  = '"\'"+textFont+"\'"';
		$replace = '(textFont.substring(0,8) == \'var(--o-\') ? textFont : "\'"+textFont+"\'"';
		$str     = str_replace( $search, $replace, $str );

		$search  = '"\'"+displayFont+"\'"';
		$replace = '(displayFont.substring(0,8) == \'var(--o-\') ? displayFont : "\'"+displayFont+"\'"';
		$str     = str_replace( $search, $replace, $str );

		// write the entire string.
		$wp_filesystem->put_contents( $target_file, $str );
	}

	/**
	 * Patch Oxygen Iframe JS
	 *
	 * @return void
	 * @since 0.3.0
	 * @since 1.11.3 Switch to WP_Filesystem.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public static function oxy_mute_iframe_js() {
		global $wp_filesystem;
		require_once ABSPATH . '/wp-admin/includes/file.php';
		WP_Filesystem();

		$target_file = WP_PLUGIN_DIR . '/oxygen/component-framework/angular/controllers/iframe.min.js';
		if ( ! $wp_filesystem->is_writable( $target_file ) ) {
			return;
		}
		// read the entire string.
		$str = $wp_filesystem->get_contents( $target_file );
		if ( str_contains( $str, 'var(--o-' ) ) {
			return;
		}

		$search  = '"\'"+t+"\'"';
		$replace = 't.substring(0,8) == \'var(--o-\' ? t : "\'" + t + "\'"';
		$str     = str_replace( $search, $replace, $str );

		$search  = '"\'"+o+"\'"';
		$replace = 'o.substring(0,8) == \'var(--o-\' ? o : "\'" + o + "\'"';
		$str     = str_replace( $search, $replace, $str );

		$search  = '"url("+m+")"';
		$replace = 'm.substring(0,8) == "var(--o-" || m.includes("gradient") ? m : "url("+m+")"';
		$str     = str_replace( $search, $replace, $str );

		// write the entire string.
		$wp_filesystem->put_contents( $target_file, $str );
	}

	/**
	 * Registers Selectors in the builder to make them available
	 * for Autosuggestion
	 *
	 * @return void
	 * @since 1.2.3
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public function register_selectors() {
		$set_to_register = $this->oxyprops_current_settings['settings']['features']['selectors'];
		if ( 'none' === $set_to_register && 'Bricks' === $this->oxyprops_builder ) {
			$this->delete_selectors();
			return;
		}
		$target_file      = WP_PLUGIN_DIR . '/oxyprops/public/json/classes-' . $set_to_register . '.json';
		$oxyprops_classes = self::read_json( $target_file );
		if ( ! is_array( $oxyprops_classes ) || count( $oxyprops_classes ) === 0 ) {
			return;
		}
		switch ( $this->oxyprops_builder ) {
			case 'Oxygen':
				$this->register_oxygen_selectors( $oxyprops_classes );
				break;
			case 'Bricks':
				$this->delete_selectors();
				$this->register_bricks_selectors( $oxyprops_classes );
				break;
			case 'Breakdance':
				$this->register_breakdance_selectors( $oxyprops_classes );
				break;
		}
	}

	/**
	 * Delete Selectors from the builder
	 *
	 * @return void
	 * @since 1.2.3
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public function delete_selectors() {
		$target_file      = WP_PLUGIN_DIR . '/oxyprops/public/json/classes-essentials.json';
		$oxyprops_classes = self::read_json( $target_file );
		if ( ! is_array( $oxyprops_classes ) || count( $oxyprops_classes ) === 0 ) {
			return;
		}
		switch ( $this->oxyprops_builder ) {
			case 'Oxygen':
				$this->delete_oxygen_selectors( $oxyprops_classes );
				break;
			case 'Bricks':
				$this->delete_bricks_selectors( $oxyprops_classes );
				break;
			case 'Breakdance':
                // phpcs:ignore
                // $this->delete_breakdance_selectors( $oxyprops_classes );
				break;
		}
	}

	/**
	 * Registers Colors in the builder
	 *
	 * @return void
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public function register_colors() {
		switch ( $this->oxyprops_builder ) {
			case 'Oxygen':
				$this->register_oxygen_colors();
				break;
			case 'Bricks':
				$this->register_bricks_colors();
				break;
			case 'Breakdance':
				$this->register_breakdance_colors();
				break;
		}
	}

	/**
	 * Registers Colors in the builder
	 *
	 * @return void
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public function delete_colors() {
		switch ( $this->oxyprops_builder ) {
			case 'Oxygen':
				$this->delete_oxygen_colors();
				break;
			case 'Bricks':
                // phpcs:ignore
                // $this->delete_bricks_colors();
				break;
			case 'Breakdance':
                // phpcs:ignore
                // $this->delete_breakdance_colors();
				break;
		}
	}

	/**
	 * Registers Selectors in Oxygen
	 *
	 * @since 1.2.3
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @param array $oxyprops_classes An array of selectors to be registered.
	 *
	 * @return void
	 */
	public function register_oxygen_selectors( $oxyprops_classes ) {
		$current_folders = (array) get_option( 'ct_style_folders', array() );
		$current_classes = get_option( 'ct_components_classes', array() );
		foreach ( $oxyprops_classes as $folder => $classes ) {
			$folder_name = 'OxyProps-' . $folder;
			if ( ! array_key_exists( $folder_name, $current_folders ) ) {
				$current_folders[ $folder_name ] = array(
					'key'    => $folder_name,
					'status' => 0,
				);
			}
			foreach ( $classes as $class ) {
				$class_definition = array(
					'key'      => $class,
					'parent'   => $folder_name,
					'original' => array(
						'selector-locked' => 'true',
					),
				);
				if (
					array_key_exists( $class, $current_classes ) &&
					is_array( $current_classes[ $class ] ) &&
					! empty( $current_classes[ $class ] )
				) {
					$class_definition = array_merge( $current_classes[ $class ], $class_definition );
				}
				$current_classes[ $class ] = $class_definition;
			}
		}
		update_option( 'ct_style_folders', $current_folders );
		update_option( 'ct_components_classes', $current_classes, get_option( 'oxygen_options_autoload' ) );
	}

	/**
	 * Deletes Selectors in Oxygen
	 *
	 * @since 1.2.3
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @param array $oxyprops_classes An array of selectors to be deleted.
	 *
	 * @return void
	 */
	public function delete_oxygen_selectors( $oxyprops_classes ) {
		$current_folders = (array) get_option( 'ct_style_folders', array() );
		$current_classes = get_option( 'ct_components_classes', array() );
		foreach ( $oxyprops_classes as $folder => $classes ) {
			$folder_name = 'OxyProps-' . $folder;

			foreach ( $current_classes as $class => $content ) {
				if ( isset( $content['parent'] ) && $folder_name === $content['parent'] ) {
					unset( $current_classes[ $class ] );
				}
			}

			if ( array_key_exists( $folder_name, $current_folders ) ) {
				unset( $current_folders[ $folder_name ] );
			}
		}
		update_option( 'ct_style_folders', $current_folders );
		update_option( 'ct_components_classes', $current_classes );
	}

	/**
	 * Registers Selectors in Bricks
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @param array $oxyprops_classes An array of selectors to be registered.
	 *
	 * @return void
	 */
	public function register_bricks_selectors( $oxyprops_classes ) {
		$bricks_current_global_classes       = (array) get_option( 'bricks_global_classes', array() );
		$bricks_current_locked_classes       = (array) get_option( 'bricks_global_classes_locked', array() );
		$bricks_current_global_classes_names = array_column( $bricks_current_global_classes, 'name' );
		$bricks_new_global_classes           = $bricks_current_global_classes;
		$bricks_new_locked_classes           = $bricks_current_locked_classes;
		foreach ( $oxyprops_classes as $oxyprops_class_category ) {
			foreach ( $oxyprops_class_category as $oxyprops_class ) {
				if ( ! in_array( $oxyprops_class, $bricks_current_global_classes_names, true ) ) {
					$bricks_new_global_classes[] = array(
						'id'       => $this->oxyprops_slug . '-' . $oxyprops_class,
						'name'     => $oxyprops_class,
						'settings' => array(),
					);
				}
				if ( ! in_array( $this->oxyprops_slug . '-' . $oxyprops_class, $bricks_current_locked_classes, true ) ) {
					$bricks_new_locked_classes[] = $this->oxyprops_slug . '-' . $oxyprops_class;
				}
			}
			update_option( 'bricks_global_classes', $bricks_new_global_classes, false );
			update_option( 'bricks_global_classes_locked', $bricks_new_locked_classes, false );
		}
	}

	/**
	 * Deletes Selectors in Oxygen
	 *
	 * @since 1.4.9
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @param array $oxyprops_classes An array of selectors to be registered.
	 *
	 * @return void
	 */
	public function delete_bricks_selectors( $oxyprops_classes ) {
		$bricks_current_global_classes = (array) get_option( 'bricks_global_classes', array() );
		$bricks_current_locked_classes = (array) get_option( 'bricks_global_classes_locked', array() );

		$bricks_new_global_classes = array();
		$bricks_new_locked_classes = array();

		foreach ( $bricks_current_global_classes as $class ) {
			$class_id = $class['id'];
			if ( ! str_contains( $class_id, $this->oxyprops_slug . '-' ) ) {
				$bricks_new_global_classes[] = $class;
			}
		}

		foreach ( $bricks_current_locked_classes as $class ) {
			if ( ! str_contains( $class, $this->oxyprops_slug . '-' ) ) {
				$bricks_new_locked_classes[] = $class;
			}
		}

		update_option( 'bricks_global_classes', $bricks_new_global_classes, false );
		update_option( 'bricks_global_classes_locked', $bricks_new_locked_classes, false );
	}

	/**
	 * Registers Selectors in Breakdance
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @param array $oxyprops_classes An array of selectors to be registered.
	 *
	 * @return void
	 */
	public function register_breakdance_selectors( $oxyprops_classes ) {
		// Empty.
	}

	/**
	 * Registers Colors in Oxygen
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function register_oxygen_colors() {
		// Empty.
	}

	/**
	 * Deletes Colors in Oxygen
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function delete_oxygen_colors() {
		// Empty.
	}

	/**
	 * Registers Colors in Bricks
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function register_bricks_colors() {
		$defaults                     = Defaults::get_instance();
		$bricks_current_color_palette = (array) get_option( 'bricks_color_palette', array() );
		$oxyprops_palettes            = $defaults->get_oxyprops_palettes();
		$oxyprops_palettes_ids        = array();
		$bricks_new_color_palette     = $bricks_current_color_palette;

		// Create OxyProps Palettes if they don't exist.
		foreach ( $oxyprops_palettes as $palette => $slug ) {
			$id = array_search( $this->oxyprops_slug . '-' . $slug, array_column( $bricks_current_color_palette, 'id' ), true );
			if ( false === $id ) {
				$bricks_new_color_palette[] = array(
					'id'      => $this->oxyprops_slug . '-' . $slug,
					'name'    => $palette,
					'colors'  => array(),
					'default' => 'logical' === $slug ? 1 : 0,
				);
				$id                         = array_key_last( $bricks_new_color_palette );
			}
			$bricks_new_color_palette[ $id ]['colors'] = array();
			$oxyprops_shades                           = $defaults->get_oxyprops_default_shades();

			// Get the corresponding colors.
			switch ( $slug ) {
				case 'logical':
					$oxyprops_colors = $defaults->get_oxyprops_logical_colors();
					foreach ( $oxyprops_colors as $color ) {
						// If the color is not present, let's add it.
						if ( ! in_array( $this->oxyprops_slug . '-' . $color, array_column( $bricks_new_color_palette[ $id ]['colors'], 'id' ), true ) ) {
							$bricks_new_color_palette[ $id ]['colors'][] = array(
								'id'   => $this->oxyprops_slug . '-' . $color,
								'name' => $color,
								'raw'  => "var(--o-{$color})",
							);
						}
					}
					break;
				case 'custom':
					$user_colors     = $this->oxyprops_current_settings['settings']['colors'];
					$oxyprops_colors = $defaults->get_oxyprops_custom_colors();
					foreach ( $oxyprops_colors as $color ) {
						foreach ( $oxyprops_shades as $shade => $value ) {
							if ( ! in_array( $this->oxyprops_slug . '-' . $color . '-' . $shade, array_column( $bricks_new_color_palette[ $id ]['colors'], 'id' ), true ) ) {
								$bricks_new_color_palette[ $id ]['colors'][] = array(
									'id'   => $this->oxyprops_slug . '-' . $color . '-' . $shade,
									'name' => $color . '-' . $shade,
									'raw'  => "var(--o-{$color}-{$shade})",
								);
							}
						}
					}
					break;
				case 'color':
					$oxyprops_colors = $defaults->get_oxyprops_default_colors();
					foreach ( $oxyprops_colors as $color ) {
						foreach ( $oxyprops_shades as $shade => $value ) {
							if ( ! in_array( $this->oxyprops_slug . '-' . $color . '-' . $shade, array_column( $bricks_new_color_palette[ $id ]['colors'], 'id' ), true ) ) {
								$bricks_new_color_palette[ $id ]['colors'][] = array(
									'id'   => $this->oxyprops_slug . '-' . $color . '-' . $shade,
									'name' => $color . '-' . $shade,
									'raw'  => "var(--o-{$color}-{$shade})",
								);
							}
						}
					}
					break;
			}
		}
		update_option( 'bricks_color_palette', $bricks_new_color_palette, false );
	}

	/**
	 * Registers Colors in Breakdance
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function register_breakdance_colors() {
		// Empty.
	}

	/**
	 * Loads a JSON file and returns an array
	 *
	 * @param string $target_file File to be read.
	 * @return array
	 * @since 1.2.3
	 * @since 1.11.3 Switch to WP_Filesystem.
	 * @author Cédric Bontems <contact@oxyprops.com>
	 */
	public static function read_json( $target_file ) {
		global $wp_filesystem;
		require_once ABSPATH . '/wp-admin/includes/file.php';
		WP_Filesystem();
		$json = json_decode( $wp_filesystem->get_contents( $target_file ), true );
		return $json;
	}

	/**
	 * Loop through the settings and
	 * create them if they don't exist.
	 *
	 * @return void
	 */
	public function register_settings() {
		$defaults         = Defaults::get_instance();
		$initial_settings = array( 'settings' => $defaults->get_oxyprops_default_settings() );
		$current_settings = get_option( $this->oxyprops_slug );
		if ( ! $current_settings ) {
			add_option( $this->oxyprops_slug, $initial_settings );
		} else {
			update_option( $this->oxyprops_slug, array_replace_recursive( $initial_settings, $current_settings ) );
		}
	}

	/**
	 * Loop through user color selection and enqueue
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return string
	 */
	public function get_user_color_choices() {
		$defaults     = Defaults::get_instance();
		$colors       = isset( $this->oxyprops_current_settings['settings']['colors'] ) ? $this->oxyprops_current_settings['settings']['colors'] : $defaults->get_oxyprops_default_settings()['colors'];
		$color_shades = $defaults->get_oxyprops_default_shades();

		$user_colors_css = ':where(html){';

		// Enqueue User Color choices.
		foreach ( $colors as $color => $def ) {
			if ( ! isset( $def['load'] ) || ! $def['load'] ) {
				continue;
			}
			// Define the reference color from its Hex code.
			$color_hsl        = self::hexToHsl( $def['hex'] );
			$user_colors_css .= "--o-{$color}-hue: {$color_hsl['h']};--o-{$color}-saturation: {$color_hsl['s']};--o-{$color}-lightness: {$color_hsl['l']};";

			$swatch = $def['swatch'];
			foreach ( $color_shades as $clean_shade => $shade ) {
				$hsl              = $swatch[ $clean_shade ];
				$user_colors_css .= "--o-{$color}-{$clean_shade}-hue: {$hsl['h']};--o-{$color}-{$clean_shade}-saturation: {$hsl['s']};--o-{$color}-{$clean_shade}-lightness: {$hsl['l']};";

				// Define hsl color weight from optimized CIELab palette.
				$user_colors_css .= "--o-{$color}-{$clean_shade}-hsl: var(--o-{$color}-{$clean_shade}-hue) var(--o-{$color}-{$clean_shade}-saturation) var(--o-{$color}-{$clean_shade}-lightness);";

				// Define color weight from hsl def.
				$user_colors_css .= "--o-{$color}-{$clean_shade}: hsl(var(--o-{$color}-{$clean_shade}-hsl));";
			}
		}

		// Define pure White & Black.
		$user_colors_css .= '--o-white: #FFF;--o-white-hsl: 0deg 0% 100%;--o-black: #000;--o-black-hsl: 0deg 0% 0%;';
		$user_colors_css .= '}';
		// Define default body text color to text-1.
		$user_colors_css .= ':where(html,body,[class*="o-sub"]){';
		$user_colors_css .= 'color:var(--o-text-2)';
		$user_colors_css .= '}';
		// Except when inside a Bricks rich text.
		if ( 'Oxygen' === $this->oxyprops_builder ) {
			$user_colors_css .= ':where(.ct-text-block){';
			$user_colors_css .= 'color:var(--o-text-2)';
			$user_colors_css .= '}';
		}
		// Except when inside a Bricks rich text.
		if ( 'Bricks' === $this->oxyprops_builder ) {
			$user_colors_css .= ':where(.brxe-text-basic,.brxe-text){';
			$user_colors_css .= 'color:var(--o-text-2)';
			$user_colors_css .= '}';
			$user_colors_css .= '.brxe-text>p{color: revert;}';
		}
		// Fix Gutenberg colors in FF and Safari.
		$user_colors_css .= '.editor-styles-wrapper {';
		$user_colors_css .= 'color:var(--o-text-2);';
		$user_colors_css .= 'font-size:var(--o-fluid-font);';
		$user_colors_css .= '}';
		// Define default Headdings color to text-1.
		$user_colors_css .= ':where(h1,h2,h3,h4,h5,h6),';
		// Fix Gutenberg colors in FF and Safari.
		$user_colors_css .= '.editor-styles-wrapper :where(h1,h2,h3,h4,h5,h6) {';
		$user_colors_css .= 'color:var(--o-text-1);';
		$user_colors_css .= 'font-size:var(--o-fluid-font);';
		$user_colors_css .= '}';

		// We a re done.
		return $user_colors_css;
	}

	/**
	 * Loop through user theme selection and enqueue
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return string
	 */
	public function get_user_theme_choices() {
		$nb_surfaces     = 5;
		$nb_texts        = 2;
		$defaults        = Defaults::get_instance();
		$logicals        = $defaults->get_oxyprops_chromatic_colors();
		$modes           = $defaults->get_oxyprops_color_modes();
		$shades          = $defaults->get_oxyprops_default_shades();
		$theme           = isset( $this->oxyprops_current_settings['settings']['theme'] ) ? $this->oxyprops_current_settings['settings']['theme'] : $defaults->get_oxyprops_default_settings()['theme'];
		$user_themes_css = ':where(html), html :where(.editor-styles-wrapper){';
		// Does user want to lock scheme?
		if ( $theme['lock_mode'] ) {
			// Dark = Light.

			// Neutral Texts.
			for ( $i = 1; $i <= $nb_texts; $i++ ) {
				$prop             = 'text-' . $i;
				$user_themes_css .= "--o-{$prop}-dark: var(--o-{$prop}-light);";
				$user_themes_css .= "--o-{$prop}-dark-hsl: var(--o-{$prop}-light-hsl);";
			}

			// Neutral Surfaces.
			for ( $i = 1; $i <= $nb_surfaces; $i++ ) {
				$prop             = 'surface-' . $i;
				$user_themes_css .= "--o-{$prop}-dark: var(--o-{$prop}-light);";
				$user_themes_css .= "--o-{$prop}-dark-hsl: var(--o-{$prop}-light-hsl);";
			}

			// Black & White.
			$user_themes_css .= '--o-bw-dark: var(--o-bw-light);--o-bw-dark-hsl: var(--o-bw-light-hsl);';
			$user_themes_css .= '--o-wb-dark: var(--o-wb-light);--o-wb-dark-hsl: var(--o-wb-light-hsl);';

			// Shadow Color & Strength.
			$user_themes_css .= '--o-shadow-color-dark: var(--o-shadow-color-light);';
			$user_themes_css .= '--o-shadow-strength-dark: var(--o-shadow-strength-light);';

			// Scrollbar.
			$user_themes_css .= '--o-scrollbar-color-dark: var(--o-scrollbar-color-light);';

			foreach ( $logicals as $color ) {
				// Colors.
				$color            = 'link' === $color ? 'links' : $color;
				$prop             = $color;
				$prop_hover       = $color . '-hover';
				$user_themes_css .= "--o-{$prop}-dark: var(--o-{$prop}-light);";
				$user_themes_css .= "--o-{$prop}-dark-hsl: var(--o-{$prop}-light-hsl);";
				$user_themes_css .= "--o-{$prop_hover}-dark: var(--o-{$prop_hover}-light);";

				// Color Texts.
				for ( $i = 1; $i <= $nb_texts; $i++ ) {
					$color            = 'link' === $color ? 'links' : $color;
					$prop             = 'text-' . $i . '-' . $color;
					$user_themes_css .= "--o-{$prop}dark: var(--o-{$prop}light);";
					$user_themes_css .= "--o-{$prop}dark-hsl: var(--o-{$prop}light-hsl);";
				}

				// Color Surfaces.
				for ( $i = 1; $i <= $nb_surfaces; $i++ ) {
					$color            = 'link' === $color ? 'links' : $color;
					$prop             = 'surface-' . $i . '-' . $color;
					$user_themes_css .= "--o-{$prop}dark: var(--o-{$prop}light);";
					$user_themes_css .= "--o-{$prop}dark-hsl: var(--o-{$prop}light-hsl);";
				}
			}
			// Only pass  light mode.
			$modes = array( 'light' );
		}

		foreach ( $modes as $mode ) {
			if ( ( 'dim' === $mode || 'dim-invert' === $mode ) && ! $theme['load_dim'] ) {
				continue;
			}
			if ( 'colorful-light' === $mode && ! $theme['load_colorful_light'] ) {
				continue;
			}
			if ( 'colorful-dark' === $mode && ! $theme['load_colorful_dark'] ) {
				continue;
			}

			$option_mode = 'dim-invert' === $mode ? 'dim' : $mode;
			$neutral     = 'colorful-light' === $mode || 'colorful-dark' === $mode ? 'brand' : 'canvas';

			// First, define our color props for the schemes.

			// Neutral Texts.
			for ( $i = 1; $i <= $nb_texts; $i++ ) {
				$option           = $theme[ 'text' . $i . '_' . str_replace( '-', '_', $option_mode ) ];
				$prop             = 'text-' . $i . '-' . $mode;
				$user_themes_css .= "--o-{$prop}: var(--o-{$neutral}-{$option});";
				$user_themes_css .= "--o-{$prop}-hsl: var(--o-{$neutral}-{$option}-hsl);";
			}

			// Neutral Surfaces.
			for ( $i = 1; $i <= $nb_surfaces; $i++ ) {
				if ( 5 > $i ) {
					$option = $theme[ 'surface' . $i . '_' . str_replace( '-', '_', $option_mode ) ];
				}

				// Define secret surface-5.
				if ( 5 === $i ) {
					$nb_shades      = count( $shades ) - 1;
					$s4             = $theme[ 'surface4_' . str_replace( '-', '_', $option_mode ) ];
					$keys           = array_keys( $shades );
					$s4_shade_index = array_search( $s4, $keys, true );
					switch ( $mode ) {
						case 'dark':
						case 'dim':
						case 'colorful-dark':
							$option = $s4_shade_index > 0 ? $shades[ $keys[ $s4_shade_index - 1 ] ] : $s4;
							break;
						default:
							$option = $s4_shade_index < $nb_shades ? $shades[ $keys[ $s4_shade_index + 1 ] ] : $s4;
					}
				}
				$prop             = 'surface-' . $i . '-' . $mode;
				$user_themes_css .= "--o-{$prop}: var(--o-{$neutral}-{$option});";
				$user_themes_css .= "--o-{$prop}-hsl: var(--o-{$neutral}-{$option}-hsl);";
			}

			// Black & White.
			switch ( $mode ) {
				case 'light':
				case 'colorful-light':
				case 'dim-invert':
					$user_themes_css .= "--o-bw-{$mode}: var(--o-black);--o-bw-{$mode}-hsl: var(--o-black-hsl);";
					$user_themes_css .= "--o-wb-{$mode}: var(--o-white);--o-wb-{$mode}-hsl: var(--o-white-hsl);";
					break;
				case 'dark':
				case 'dim':
				case 'colorful-dark':
					$user_themes_css .= "--o-bw-{$mode}: var(--o-white);--o-bw-{$mode}-hsl: var(--o-white-hsl);";
					$user_themes_css .= "--o-wb-{$mode}: var(--o-black);--o-wb-{$mode}-hsl: var(--o-black-hsl);";
					break;
			}

			// Shadow Color & Strength.
			$user_themes_css .= "--o-shadow-color-{$mode}: var(--o-canvas-{$theme['shadow_color_' . str_replace('-', '_',$option_mode)]}-hsl);";
			$user_themes_css .= "--o-shadow-strength-{$mode}: {$theme['shadow_strength_' . str_replace('-', '_',$option_mode)]}%;";

			// Scrollbar.
			$user_themes_css .= "--o-scrollbar-color-{$mode}: var(--o-canvas-{$theme['scrollbar_' . str_replace('-', '_',$option_mode)]});";

			foreach ( $logicals as $color ) {
				// Colors.
				$option           = $theme[ $color . '_' . str_replace( '-', '_', $option_mode ) ];
				$option_hover     = $theme[ $color . '_' . str_replace( '-', '_', $option_mode ) . '_hover' ];
				$color            = 'link' === $color ? 'links' : $color;
				$prop             = $color . '-' . $mode;
				$prop_hover       = $color . '-hover-' . $mode;
				$user_themes_css .= "--o-{$prop}: var(--o-{$color}-{$option});";
				$user_themes_css .= "--o-{$prop}-hsl: var(--o-{$color}-{$option}-hsl);";
				$user_themes_css .= "--o-{$prop_hover}: var(--o-{$color}-{$option_hover});";

				// Color Texts.
				for ( $i = 1; $i <= $nb_texts; $i++ ) {
					$option           = $theme[ 'text' . $i . '_' . str_replace( '-', '_', $option_mode ) ];
					$color            = 'link' === $color ? 'links' : $color;
					$prop             = 'text-' . $i . '-' . $color . '-' . $mode;
					$user_themes_css .= "--o-{$prop}: var(--o-{$color}-{$option});";
					$user_themes_css .= "--o-{$prop}-hsl: var(--o-{$color}-{$option}-hsl);";
				}

				// Color Surfaces.
				for ( $i = 1; $i <= $nb_surfaces - 1; $i++ ) {
					$option           = $theme[ 'surface' . $i . '_' . str_replace( '-', '_', $option_mode ) ];
					$color            = 'link' === $color ? 'links' : $color;
					$prop             = 'surface-' . $i . '-' . $color . '-' . $mode;
					$user_themes_css .= "--o-{$prop}: var(--o-{$color}-{$option});";
					$user_themes_css .= "--o-{$prop}-hsl: var(--o-{$color}-{$option}-hsl);";
				}
			}
		}
		// Set default background surface.
		$user_themes_css .= 'background-color: var(--o-surface-1);';
		$user_themes_css .= '}';

		// Now, let's generate actual schemes.

		// Light first.
		$light_theme      = 'color-scheme: light;--o-brand: var(--o-brand-light);--o-brand-hsl: var(--o-brand-light-hsl);--o-brand-hover: var(--o-brand-hover-light);--o-accent: var(--o-accent-light);--o-accent-hsl: var(--o-accent-light-hsl);--o-accent-hover: var(--o-accent-hover-light);--o-links: var(--o-links-light);--o-links-hsl: var(--o-links-light-hsl);--o-links-hover: var(--o-links-hover-light);--o-visited: var(--o-visited-light);--o-visited-hsl: var(--o-visited-light-hsl);--o-visited-hover: var(--o-visited-hover-light);--o-text-1: var(--o-text-1-light);--o-text-2: var(--o-text-2-light);--o-surface-1: var(--o-surface-1-light);--o-surface-2: var(--o-surface-2-light);--o-surface-3: var(--o-surface-3-light);--o-surface-4: var(--o-surface-4-light);--o-surface-5: var(--o-surface-5-light);--o-text-1-hsl: var(--o-text-1-light-hsl);--o-text-2-hsl: var(--o-text-2-light-hsl);--o-surface-1-hsl: var(--o-surface-1-light-hsl);--o-surface-2-hsl: var(--o-surface-2-light-hsl);--o-surface-3-hsl: var(--o-surface-3-light-hsl);--o-surface-4-hsl: var(--o-surface-4-light-hsl);--o-text-1-brand: var(--o-text-1-brand-light);--o-text-2-brand: var(--o-text-2-brand-light);--o-surface-1-brand: var(--o-surface-1-brand-light);--o-surface-2-brand: var(--o-surface-2-brand-light);--o-surface-3-brand: var(--o-surface-3-brand-light);--o-surface-4-brand: var(--o-surface-4-brand-light);--o-text-1-brand-hsl: var(--o-text-1-brand-light-hsl);--o-text-2-brand-hsl: var(--o-text-2-brand-light-hsl);--o-surface-1-brand-hsl: var(--o-surface-1-brand-light-hsl);--o-surface-2-brand-hsl: var(--o-surface-2-brand-light-hsl);--o-surface-3-brand-hsl: var(--o-surface-3-brand-light-hsl);--o-surface-4-brand-hsl: var(--o-surface-4-brand-light-hsl);--o-text-1-accent: var(--o-text-1-accent-light);--o-text-2-accent: var(--o-text-2-accent-light);--o-surface-1-accent: var(--o-surface-1-accent-light);--o-surface-2-accent: var(--o-surface-2-accent-light);--o-surface-3-accent: var(--o-surface-3-accent-light);--o-surface-4-accent: var(--o-surface-4-accent-light);--o-text-1-accent-hsl: var(--o-text-1-accent-light-hsl);--o-text-2-accent-hsl: var(--o-text-2-accent-light-hsl);--o-surface-1-accent-hsl: var(--o-surface-1-accent-light-hsl);--o-surface-2-accent-hsl: var(--o-surface-2-accent-light-hsl);--o-surface-3-accent-hsl: var(--o-surface-3-accent-light-hsl);--o-surface-4-accent-hsl: var(--o-surface-4-accent-light-hsl);--o-text-1-links: var(--o-text-1-links-light);--o-text-2-links: var(--o-text-2-links-light);--o-surface-1-links: var(--o-surface-1-links-light);--o-surface-2-links: var(--o-surface-2-links-light);--o-surface-3-links: var(--o-surface-3-links-light);--o-surface-4-links: var(--o-surface-4-links-light);--o-text-1-links-hsl: var(--o-text-1-links-light-hsl);--o-text-2-links-hsl: var(--o-text-2-links-light-hsl);--o-surface-1-links-hsl: var(--o-surface-1-links-light-hsl);--o-surface-2-links-hsl: var(--o-surface-2-links-light-hsl);--o-surface-3-links-hsl: var(--o-surface-3-links-light-hsl);--o-surface-4-links-hsl: var(--o-surface-4-links-light-hsl);--o-text-1-visited: var(--o-text-1-visited-light);--o-text-2-visited: var(--o-text-2-visited-light);--o-surface-1-visited: var(--o-surface-1-visited-light);--o-surface-2-visited: var(--o-surface-2-visited-light);--o-surface-3-visited: var(--o-surface-3-visited-light);--o-surface-4-visited: var(--o-surface-4-visited-light);--o-text-1-visited-hsl: var(--o-text-1-visited-light-hsl);--o-text-2-visited-hsl: var(--o-text-2-visited-light-hsl);--o-surface-1-visited-hsl: var(--o-surface-1-visited-light-hsl);--o-surface-2-visited-hsl: var(--o-surface-2-visited-light-hsl);--o-surface-3-visited-hsl: var(--o-surface-3-visited-light-hsl);--o-surface-4-visited-hsl: var(--o-surface-4-visited-light-hsl);--o-bw: var(--o-bw-light);--o-bw-hsl: var(--o-bw-light-hsl);--o-wb: var(--o-wb-light);--o-wb-hsl: var(--o-wb-light-hsl);--o-shadow-color: var(--o-shadow-color-light);--o-shadow-strength: var(--o-shadow-strength-light);--o-scrollbar-color: var(--o-scrollbar-color-light);';
		$user_themes_css .= ':root{' . $light_theme . '}';

		if ( ! $theme['lock_mode'] && $theme['load_dark'] ) {

			$dark_theme       = 'color-scheme: dark;--o-brand: var(--o-brand-dark);--o-brand-hsl: var(--o-brand-dark-hsl);--o-brand-hover: var(--o-brand-hover-dark);--o-accent: var(--o-accent-dark);--o-accent-hsl: var(--o-accent-dark-hsl);--o-accent-hover: var(--o-accent-hover-dark);--o-links: var(--o-links-dark);--o-links-hsl: var(--o-links-dark-hsl);--o-links-hover: var(--o-links-hover-dark);--o-visited: var(--o-visited-dark);--o-visited-hsl: var(--o-visited-dark-hsl);--o-visited-hover: var(--o-visited-hover-dark);--o-text-1: var(--o-text-1-dark);--o-text-2: var(--o-text-2-dark);--o-surface-1: var(--o-surface-1-dark);--o-surface-2: var(--o-surface-2-dark);--o-surface-3: var(--o-surface-3-dark);--o-surface-4: var(--o-surface-4-dark);--o-surface-5: var(--o-surface-5-dark);--o-text-1-hsl: var(--o-text-1-dark-hsl);--o-text-2-hsl: var(--o-text-2-dark-hsl);--o-surface-1-hsl: var(--o-surface-1-dark-hsl);--o-surface-2-hsl: var(--o-surface-2-dark-hsl);--o-surface-3-hsl: var(--o-surface-3-dark-hsl);--o-surface-4-hsl: var(--o-surface-4-dark-hsl);--o-text-1-brand: var(--o-text-1-brand-dark);--o-text-2-brand: var(--o-text-2-brand-dark);--o-surface-1-brand: var(--o-surface-1-brand-dark);--o-surface-2-brand: var(--o-surface-2-brand-dark);--o-surface-3-brand: var(--o-surface-3-brand-dark);--o-surface-4-brand: var(--o-surface-4-brand-dark);--o-text-1-brand-hsl: var(--o-text-1-brand-dark-hsl);--o-text-2-brand-hsl: var(--o-text-2-brand-dark-hsl);--o-surface-1-brand-hsl: var(--o-surface-1-brand-dark-hsl);--o-surface-2-brand-hsl: var(--o-surface-2-brand-dark-hsl);--o-surface-3-brand-hsl: var(--o-surface-3-brand-dark-hsl);--o-surface-4-brand-hsl: var(--o-surface-4-brand-dark-hsl);--o-text-1-accent: var(--o-text-1-accent-dark);--o-text-2-accent: var(--o-text-2-accent-dark);--o-surface-1-accent: var(--o-surface-1-accent-dark);--o-surface-2-accent: var(--o-surface-2-accent-dark);--o-surface-3-accent: var(--o-surface-3-accent-dark);--o-surface-4-accent: var(--o-surface-4-accent-dark);--o-text-1-accent-hsl: var(--o-text-1-accent-dark-hsl);--o-text-2-accent-hsl: var(--o-text-2-accent-dark-hsl);--o-surface-1-accent-hsl: var(--o-surface-1-accent-dark-hsl);--o-surface-2-accent-hsl: var(--o-surface-2-accent-dark-hsl);--o-surface-3-accent-hsl: var(--o-surface-3-accent-dark-hsl);--o-surface-4-accent-hsl: var(--o-surface-4-accent-dark-hsl);--o-text-1-links: var(--o-text-1-links-dark);--o-text-2-links: var(--o-text-2-links-dark);--o-surface-1-links: var(--o-surface-1-links-dark);--o-surface-2-links: var(--o-surface-2-links-dark);--o-surface-3-links: var(--o-surface-3-links-dark);--o-surface-4-links: var(--o-surface-4-links-dark);--o-text-1-links-hsl: var(--o-text-1-links-dark-hsl);--o-text-2-links-hsl: var(--o-text-2-links-dark-hsl);--o-surface-1-links-hsl: var(--o-surface-1-links-dark-hsl);--o-surface-2-links-hsl: var(--o-surface-2-links-dark-hsl);--o-surface-3-links-hsl: var(--o-surface-3-links-dark-hsl);--o-surface-4-links-hsl: var(--o-surface-4-links-dark-hsl);--o-text-1-visited: var(--o-text-1-visited-dark);--o-text-2-visited: var(--o-text-2-visited-dark);--o-surface-1-visited: var(--o-surface-1-visited-dark);--o-surface-2-visited: var(--o-surface-2-visited-dark);--o-surface-3-visited: var(--o-surface-3-visited-dark);--o-surface-4-visited: var(--o-surface-4-visited-dark);--o-text-1-visited-hsl: var(--o-text-1-visited-dark-hsl);--o-text-2-visited-hsl: var(--o-text-2-visited-dark-hsl);--o-surface-1-visited-hsl: var(--o-surface-1-visited-dark-hsl);--o-surface-2-visited-hsl: var(--o-surface-2-visited-dark-hsl);--o-surface-3-visited-hsl: var(--o-surface-3-visited-dark-hsl);--o-surface-4-visited-hsl: var(--o-surface-4-visited-dark-hsl);--o-bw: var(--o-bw-dark);--o-bw-hsl: var(--o-bw-dark-hsl);--o-wb: var(--o-wb-dark);--o-wb-hsl: var(--o-wb-dark-hsl);--o-shadow-color: var(--o-shadow-color-dark);--o-shadow-strength: var(--o-shadow-strength-dark);--o-scrollbar-color: var(--o-scrollbar-color-dark);';
			$user_themes_css .= '.o-invert {' . $dark_theme . '}';
			$user_themes_css .= '@media (prefers-color-scheme: dark){:root{' . $dark_theme . '}.o-invert {' . $light_theme . '}}';
			$user_themes_css .= '[color-scheme="light"],[color-scheme="dark"] .o-invert {' . $light_theme . '}[color-scheme="dark"], [color-scheme="light"] .o-invert {' . $dark_theme . '}';
		}

		if ( ! $theme['lock_mode'] && $theme['load_dim'] ) {
			$dim_theme        = '--o-brand: var(--o-brand-dim);--o-brand-hsl: var(--o-brand-dim-hsl);--o-brand-hover: var(--o-brand-hover-dim);--o-accent: var(--o-accent-dim);--o-accent-hsl: var(--o-accent-dim-hsl);--o-accent-hover: var(--o-accent-hover-dim);--o-links: var(--o-links-dim);--o-links-hsl: var(--o-links-dim-hsl);--o-links-hover: var(--o-links-hover-dim);--o-visited: var(--o-visited-dim);--o-visited-hsl: var(--o-visited-dim-hsl);--o-visited-hover: var(--o-visited-hover-dim);--o-text-1: var(--o-text-1-dim);--o-text-2: var(--o-text-2-dim);--o-surface-1: var(--o-surface-1-dim);--o-surface-2: var(--o-surface-2-dim);--o-surface-3: var(--o-surface-3-dim);--o-surface-4: var(--o-surface-4-dim);--o-text-1-hsl: var(--o-text-1-dim-hsl);--o-text-2-hsl: var(--o-text-2-dim-hsl);--o-surface-1-hsl: var(--o-surface-1-dim-hsl);--o-surface-2-hsl: var(--o-surface-2-dim-hsl);--o-surface-3-hsl: var(--o-surface-3-dim-hsl);--o-surface-4-hsl: var(--o-surface-4-dim-hsl);--o-text-1-brand: var(--o-text-1-brand-dim);--o-text-2-brand: var(--o-text-2-brand-dim);--o-surface-1-brand: var(--o-surface-1-brand-dim);--o-surface-2-brand: var(--o-surface-2-brand-dim);--o-surface-3-brand: var(--o-surface-3-brand-dim);--o-surface-4-brand: var(--o-surface-4-brand-dim);--o-text-1-brand-hsl: var(--o-text-1-brand-dim-hsl);--o-text-2-brand-hsl: var(--o-text-2-brand-dim-hsl);--o-surface-1-brand-hsl: var(--o-surface-1-brand-dim-hsl);--o-surface-2-brand-hsl: var(--o-surface-2-brand-dim-hsl);--o-surface-3-brand-hsl: var(--o-surface-3-brand-dim-hsl);--o-surface-4-brand-hsl: var(--o-surface-4-brand-dim-hsl);--o-text-1-accent: var(--o-text-1-accent-dim);--o-text-2-accent: var(--o-text-2-accent-dim);--o-surface-1-accent: var(--o-surface-1-accent-dim);--o-surface-2-accent: var(--o-surface-2-accent-dim);--o-surface-3-accent: var(--o-surface-3-accent-dim);--o-surface-4-accent: var(--o-surface-4-accent-dim);--o-text-1-accent-hsl: var(--o-text-1-accent-dim-hsl);--o-text-2-accent-hsl: var(--o-text-2-accent-dim-hsl);--o-surface-1-accent-hsl: var(--o-surface-1-accent-dim-hsl);--o-surface-2-accent-hsl: var(--o-surface-2-accent-dim-hsl);--o-surface-3-accent-hsl: var(--o-surface-3-accent-dim-hsl);--o-surface-4-accent-hsl: var(--o-surface-4-accent-dim-hsl);--o-text-1-links: var(--o-text-1-links-dim);--o-text-2-links: var(--o-text-2-links-dim);--o-surface-1-links: var(--o-surface-1-links-dim);--o-surface-2-links: var(--o-surface-2-links-dim);--o-surface-3-links: var(--o-surface-3-links-dim);--o-surface-4-links: var(--o-surface-4-links-dim);--o-text-1-links-hsl: var(--o-text-1-links-dim-hsl);--o-text-2-links-hsl: var(--o-text-2-links-dim-hsl);--o-surface-1-links-hsl: var(--o-surface-1-links-dim-hsl);--o-surface-2-links-hsl: var(--o-surface-2-links-dim-hsl);--o-surface-3-links-hsl: var(--o-surface-3-links-dim-hsl);--o-surface-4-links-hsl: var(--o-surface-4-links-dim-hsl);--o-text-1-visited: var(--o-text-1-visited-dim);--o-text-2-visited: var(--o-text-2-visited-dim);--o-surface-1-visited: var(--o-surface-1-visited-dim);--o-surface-2-visited: var(--o-surface-2-visited-dim);--o-surface-3-visited: var(--o-surface-3-visited-dim);--o-surface-4-visited: var(--o-surface-4-visited-dim);--o-text-1-visited-hsl: var(--o-text-1-visited-dim-hsl);--o-text-2-visited-hsl: var(--o-text-2-visited-dim-hsl);--o-surface-1-visited-hsl: var(--o-surface-1-visited-dim-hsl);--o-surface-2-visited-hsl: var(--o-surface-2-visited-dim-hsl);--o-surface-3-visited-hsl: var(--o-surface-3-visited-dim-hsl);--o-surface-4-visited-hsl: var(--o-surface-4-visited-dim-hsl);--o-bw: var(--o-bw-dim);--o-bw-hsl: var(--o-bw-dim-hsl);--o-wb: var(--o-wb-dim);--o-wb-hsl: var(--o-wb-dim-hsl);--o-shadow-color: var(--o-shadow-color-dim);--o-shadow-strength: var(--o-shadow-strength-dim);--o-scrollbar-color: var(--o-scrollbar-color-dim);';
			$dim_invert_theme = '--o-brand: var(--o-brand-dim-invert);--o-brand-hsl: var(--o-brand-dim-invert-hsl);--o-brand-hover: var(--o-brand-hover-dim-invert);--o-accent: var(--o-accent-dim-invert);--o-accent-hsl: var(--o-accent-dim-invert-hsl);--o-accent-hover: var(--o-accent-hover-dim-invert);--o-links: var(--o-links-dim-invert);--o-links-hsl: var(--o-links-dim-invert-hsl);--o-links-hover: var(--o-links-hover-dim-invert);--o-visited: var(--o-visited-dim-invert);--o-visited-hsl: var(--o-visited-dim-invert-hsl);--o-visited-hover: var(--o-visited-hover-dim-invert);--o-text-1: var(--o-text-1-dim-invert);--o-text-2: var(--o-text-2-dim-invert);--o-surface-1: var(--o-surface-1-dim-invert);--o-surface-2: var(--o-surface-2-dim-invert);--o-surface-3: var(--o-surface-3-dim-invert);--o-surface-4: var(--o-surface-4-dim-invert);--o-text-1-hsl: var(--o-text-1-dim-invert-hsl);--o-text-2-hsl: var(--o-text-2-dim-invert-hsl);--o-surface-1-hsl: var(--o-surface-1-dim-invert-hsl);--o-surface-2-hsl: var(--o-surface-2-dim-invert-hsl);--o-surface-3-hsl: var(--o-surface-3-dim-invert-hsl);--o-surface-4-hsl: var(--o-surface-4-dim-invert-hsl);--o-text-1-brand: var(--o-text-1-brand-dim-invert);--o-text-2-brand: var(--o-text-2-brand-dim-invert);--o-surface-1-brand: var(--o-surface-1-brand-dim-invert);--o-surface-2-brand: var(--o-surface-2-brand-dim-invert);--o-surface-3-brand: var(--o-surface-3-brand-dim-invert);--o-surface-4-brand: var(--o-surface-4-brand-dim-invert);--o-text-1-brand-hsl: var(--o-text-1-brand-dim-invert-hsl);--o-text-2-brand-hsl: var(--o-text-2-brand-dim-invert-hsl);--o-surface-1-brand-hsl: var(--o-surface-1-brand-dim-invert-hsl);--o-surface-2-brand-hsl: var(--o-surface-2-brand-dim-invert-hsl);--o-surface-3-brand-hsl: var(--o-surface-3-brand-dim-invert-hsl);--o-surface-4-brand-hsl: var(--o-surface-4-brand-dim-invert-hsl);--o-text-1-accent: var(--o-text-1-accent-dim-invert);--o-text-2-accent: var(--o-text-2-accent-dim-invert);--o-surface-1-accent: var(--o-surface-1-accent-dim-invert);--o-surface-2-accent: var(--o-surface-2-accent-dim-invert);--o-surface-3-accent: var(--o-surface-3-accent-dim-invert);--o-surface-4-accent: var(--o-surface-4-accent-dim-invert);--o-text-1-accent-hsl: var(--o-text-1-accent-dim-invert-hsl);--o-text-2-accent-hsl: var(--o-text-2-accent-dim-invert-hsl);--o-surface-1-accent-hsl: var(--o-surface-1-accent-dim-invert-hsl);--o-surface-2-accent-hsl: var(--o-surface-2-accent-dim-invert-hsl);--o-surface-3-accent-hsl: var(--o-surface-3-accent-dim-invert-hsl);--o-surface-4-accent-hsl: var(--o-surface-4-accent-dim-invert-hsl);--o-text-1-links: var(--o-text-1-links-dim-invert);--o-text-2-links: var(--o-text-2-links-dim-invert);--o-surface-1-links: var(--o-surface-1-links-dim-invert);--o-surface-2-links: var(--o-surface-2-links-dim-invert);--o-surface-3-links: var(--o-surface-3-links-dim-invert);--o-surface-4-links: var(--o-surface-4-links-dim-invert);--o-text-1-links-hsl: var(--o-text-1-links-dim-invert-hsl);--o-text-2-links-hsl: var(--o-text-2-links-dim-invert-hsl);--o-surface-1-links-hsl: var(--o-surface-1-links-dim-invert-hsl);--o-surface-2-links-hsl: var(--o-surface-2-links-dim-invert-hsl);--o-surface-3-links-hsl: var(--o-surface-3-links-dim-invert-hsl);--o-surface-4-links-hsl: var(--o-surface-4-links-dim-invert-hsl);--o-text-1-visited: var(--o-text-1-visited-dim-invert);--o-text-2-visited: var(--o-text-2-visited-dim-invert);--o-surface-1-visited: var(--o-surface-1-visited-dim-invert);--o-surface-2-visited: var(--o-surface-2-visited-dim-invert);--o-surface-3-visited: var(--o-surface-3-visited-dim-invert);--o-surface-4-visited: var(--o-surface-4-visited-dim-invert);--o-text-1-visited-hsl: var(--o-text-1-visited-dim-invert-hsl);--o-text-2-visited-hsl: var(--o-text-2-visited-dim-invert-hsl);--o-surface-1-visited-hsl: var(--o-surface-1-visited-dim-invert-hsl);--o-surface-2-visited-hsl: var(--o-surface-2-visited-dim-invert-hsl);--o-surface-3-visited-hsl: var(--o-surface-3-visited-dim-invert-hsl);--o-surface-4-visited-hsl: var(--o-surface-4-visited-dim-invert-hsl);--o-bw: var(--o-bw-dim-invert);--o-bw-hsl: var(--o-bw-dim-invert-hsl);--o-wb: var(--o-wb-dim-invert);--o-wb-hsl: var(--o-wb-dim-invert-hsl);--o-shadow-color: var(--o-shadow-color-dim-invert);--o-shadow-strength: var(--o-shadow-strength-dim-invert);--o-scrollbar-color: var(--o-scrollbar-color-dim-invert);';
			$user_themes_css .= '[color-scheme="dim"] {' . $dim_theme . '}[color-scheme="dim"] .o-invert {' . $dim_invert_theme . '}';
			$user_themes_css .= '@media (prefers-color-scheme: dark){[color-scheme="dim"] {' . $dim_invert_theme . '}[color-scheme="dim"] .o-invert {' . $dim_theme . '}}';
		}

		if ( ! $theme['lock_mode'] && $theme['load_colorful_light'] ) {
			$colorful_light_theme = 'color-scheme: light;--o-brand: var(--o-brand-colorful-light);--o-brand-hsl: var(--o-brand-colorful-light-hsl);--o-brand-hover: var(--o-brand-hover-colorful-light);--o-accent: var(--o-accent-colorful-light);--o-accent-hsl: var(--o-accent-colorful-light-hsl);--o-accent-hover: var(--o-accent-hover-colorful-light);--o-links: var(--o-links-colorful-light);--o-links-hsl: var(--o-links-colorful-light-hsl);--o-links-hover: var(--o-links-hover-colorful-light);--o-visited: var(--o-visited-colorful-light);--o-visited-hsl: var(--o-visited-colorful-light-hsl);--o-visited-hover: var(--o-visited-hover-colorful-light);--o-text-1: var(--o-text-1-colorful-light);--o-text-2: var(--o-text-2-colorful-light);--o-surface-1: var(--o-surface-1-colorful-light);--o-surface-2: var(--o-surface-2-colorful-light);--o-surface-3: var(--o-surface-3-colorful-light);--o-surface-4: var(--o-surface-4-colorful-light);--o-text-1-hsl: var(--o-text-1-colorful-light-hsl);--o-text-2-hsl: var(--o-text-2-colorful-light-hsl);--o-surface-1-hsl: var(--o-surface-1-colorful-light-hsl);--o-surface-2-hsl: var(--o-surface-2-colorful-light-hsl);--o-surface-3-hsl: var(--o-surface-3-colorful-light-hsl);--o-surface-4-hsl: var(--o-surface-4-colorful-light-hsl);--o-text-1-brand: var(--o-text-1-brand-colorful-light);--o-text-2-brand: var(--o-text-2-brand-colorful-light);--o-surface-1-brand: var(--o-surface-1-brand-colorful-light);--o-surface-2-brand: var(--o-surface-2-brand-colorful-light);--o-surface-3-brand: var(--o-surface-3-brand-colorful-light);--o-surface-4-brand: var(--o-surface-4-brand-colorful-light);--o-text-1-brand-hsl: var(--o-text-1-brand-colorful-light-hsl);--o-text-2-brand-hsl: var(--o-text-2-brand-colorful-light-hsl);--o-surface-1-brand-hsl: var(--o-surface-1-brand-colorful-light-hsl);--o-surface-2-brand-hsl: var(--o-surface-2-brand-colorful-light-hsl);--o-surface-3-brand-hsl: var(--o-surface-3-brand-colorful-light-hsl);--o-surface-4-brand-hsl: var(--o-surface-4-brand-colorful-light-hsl);--o-text-1-accent: var(--o-text-1-accent-colorful-light);--o-text-2-accent: var(--o-text-2-accent-colorful-light);--o-surface-1-accent: var(--o-surface-1-accent-colorful-light);--o-surface-2-accent: var(--o-surface-2-accent-colorful-light);--o-surface-3-accent: var(--o-surface-3-accent-colorful-light);--o-surface-4-accent: var(--o-surface-4-accent-colorful-light);--o-text-1-accent-hsl: var(--o-text-1-accent-colorful-light-hsl);--o-text-2-accent-hsl: var(--o-text-2-accent-colorful-light-hsl);--o-surface-1-accent-hsl: var(--o-surface-1-accent-colorful-light-hsl);--o-surface-2-accent-hsl: var(--o-surface-2-accent-colorful-light-hsl);--o-surface-3-accent-hsl: var(--o-surface-3-accent-colorful-light-hsl);--o-surface-4-accent-hsl: var(--o-surface-4-accent-colorful-light-hsl);--o-text-1-links: var(--o-text-1-links-colorful-light);--o-text-2-links: var(--o-text-2-links-colorful-light);--o-surface-1-links: var(--o-surface-1-links-colorful-light);--o-surface-2-links: var(--o-surface-2-links-colorful-light);--o-surface-3-links: var(--o-surface-3-links-colorful-light);--o-surface-4-links: var(--o-surface-4-links-colorful-light);--o-text-1-links-hsl: var(--o-text-1-links-colorful-light-hsl);--o-text-2-links-hsl: var(--o-text-2-links-colorful-light-hsl);--o-surface-1-links-hsl: var(--o-surface-1-links-colorful-light-hsl);--o-surface-2-links-hsl: var(--o-surface-2-links-colorful-light-hsl);--o-surface-3-links-hsl: var(--o-surface-3-links-colorful-light-hsl);--o-surface-4-links-hsl: var(--o-surface-4-links-colorful-light-hsl);--o-text-1-visited: var(--o-text-1-visited-colorful-light);--o-text-2-visited: var(--o-text-2-visited-colorful-light);--o-surface-1-visited: var(--o-surface-1-visited-colorful-light);--o-surface-2-visited: var(--o-surface-2-visited-colorful-light);--o-surface-3-visited: var(--o-surface-3-visited-colorful-light);--o-surface-4-visited: var(--o-surface-4-visited-colorful-light);--o-text-1-visited-hsl: var(--o-text-1-visited-colorful-light-hsl);--o-text-2-visited-hsl: var(--o-text-2-visited-colorful-light-hsl);--o-surface-1-visited-hsl: var(--o-surface-1-visited-colorful-light-hsl);--o-surface-2-visited-hsl: var(--o-surface-2-visited-colorful-light-hsl);--o-surface-3-visited-hsl: var(--o-surface-3-visited-colorful-light-hsl);--o-surface-4-visited-hsl: var(--o-surface-4-visited-colorful-light-hsl);--o-bw: var(--o-bw-colorful-light);--o-bw-hsl: var(--o-bw-colorful-light-hsl);--o-wb: var(--o-wb-colorful-light);--o-wb-hsl: var(--o-wb-colorful-light-hsl);--o-shadow-color: var(--o-shadow-color-colorful-light);--o-shadow-strength: var(--o-shadow-strength-colorful-light);--o-scrollbar-color: var(--o-scrollbar-color-colorful-light);';
			$user_themes_css     .= '[color-scheme="colorful"],[color-scheme="light"] [color-scheme="colorful"],[color-scheme="colorful-light"],[color-scheme="colorful-dark"] .o-invert {' . $colorful_light_theme . '}';
		}

		if ( ! $theme['lock_mode'] && $theme['load_colorful_dark'] ) {
			$colorful_dark_theme = 'color-scheme: light;--o-brand: var(--o-brand-colorful-dark);--o-brand-hsl: var(--o-brand-colorful-dark-hsl);--o-brand-hover: var(--o-brand-hover-colorful-dark);--o-accent: var(--o-accent-colorful-dark);--o-accent-hsl: var(--o-accent-colorful-dark-hsl);--o-accent-hover: var(--o-accent-hover-colorful-dark);--o-links: var(--o-links-colorful-dark);--o-links-hsl: var(--o-links-colorful-dark-hsl);--o-links-hover: var(--o-links-hover-colorful-dark);--o-visited: var(--o-visited-colorful-dark);--o-visited-hsl: var(--o-visited-colorful-dark-hsl);--o-visited-hover: var(--o-visited-hover-colorful-dark);--o-text-1: var(--o-text-1-colorful-dark);--o-text-2: var(--o-text-2-colorful-dark);--o-surface-1: var(--o-surface-1-colorful-dark);--o-surface-2: var(--o-surface-2-colorful-dark);--o-surface-3: var(--o-surface-3-colorful-dark);--o-surface-4: var(--o-surface-4-colorful-dark);--o-text-1-hsl: var(--o-text-1-colorful-dark-hsl);--o-text-2-hsl: var(--o-text-2-colorful-dark-hsl);--o-surface-1-hsl: var(--o-surface-1-colorful-dark-hsl);--o-surface-2-hsl: var(--o-surface-2-colorful-dark-hsl);--o-surface-3-hsl: var(--o-surface-3-colorful-dark-hsl);--o-surface-4-hsl: var(--o-surface-4-colorful-dark-hsl);--o-text-1-brand: var(--o-text-1-brand-colorful-dark);--o-text-2-brand: var(--o-text-2-brand-colorful-dark);--o-surface-1-brand: var(--o-surface-1-brand-colorful-dark);--o-surface-2-brand: var(--o-surface-2-brand-colorful-dark);--o-surface-3-brand: var(--o-surface-3-brand-colorful-dark);--o-surface-4-brand: var(--o-surface-4-brand-colorful-dark);--o-text-1-brand-hsl: var(--o-text-1-brand-colorful-dark-hsl);--o-text-2-brand-hsl: var(--o-text-2-brand-colorful-dark-hsl);--o-surface-1-brand-hsl: var(--o-surface-1-brand-colorful-dark-hsl);--o-surface-2-brand-hsl: var(--o-surface-2-brand-colorful-dark-hsl);--o-surface-3-brand-hsl: var(--o-surface-3-brand-colorful-dark-hsl);--o-surface-4-brand-hsl: var(--o-surface-4-brand-colorful-dark-hsl);--o-text-1-accent: var(--o-text-1-accent-colorful-dark);--o-text-2-accent: var(--o-text-2-accent-colorful-dark);--o-surface-1-accent: var(--o-surface-1-accent-colorful-dark);--o-surface-2-accent: var(--o-surface-2-accent-colorful-dark);--o-surface-3-accent: var(--o-surface-3-accent-colorful-dark);--o-surface-4-accent: var(--o-surface-4-accent-colorful-dark);--o-text-1-accent-hsl: var(--o-text-1-accent-colorful-dark-hsl);--o-text-2-accent-hsl: var(--o-text-2-accent-colorful-dark-hsl);--o-surface-1-accent-hsl: var(--o-surface-1-accent-colorful-dark-hsl);--o-surface-2-accent-hsl: var(--o-surface-2-accent-colorful-dark-hsl);--o-surface-3-accent-hsl: var(--o-surface-3-accent-colorful-dark-hsl);--o-surface-4-accent-hsl: var(--o-surface-4-accent-colorful-dark-hsl);--o-text-1-links: var(--o-text-1-links-colorful-dark);--o-text-2-links: var(--o-text-2-links-colorful-dark);--o-surface-1-links: var(--o-surface-1-links-colorful-dark);--o-surface-2-links: var(--o-surface-2-links-colorful-dark);--o-surface-3-links: var(--o-surface-3-links-colorful-dark);--o-surface-4-links: var(--o-surface-4-links-colorful-dark);--o-text-1-links-hsl: var(--o-text-1-links-colorful-dark-hsl);--o-text-2-links-hsl: var(--o-text-2-links-colorful-dark-hsl);--o-surface-1-links-hsl: var(--o-surface-1-links-colorful-dark-hsl);--o-surface-2-links-hsl: var(--o-surface-2-links-colorful-dark-hsl);--o-surface-3-links-hsl: var(--o-surface-3-links-colorful-dark-hsl);--o-surface-4-links-hsl: var(--o-surface-4-links-colorful-dark-hsl);--o-text-1-visited: var(--o-text-1-visited-colorful-dark);--o-text-2-visited: var(--o-text-2-visited-colorful-dark);--o-surface-1-visited: var(--o-surface-1-visited-colorful-dark);--o-surface-2-visited: var(--o-surface-2-visited-colorful-dark);--o-surface-3-visited: var(--o-surface-3-visited-colorful-dark);--o-surface-4-visited: var(--o-surface-4-visited-colorful-dark);--o-text-1-visited-hsl: var(--o-text-1-visited-colorful-dark-hsl);--o-text-2-visited-hsl: var(--o-text-2-visited-colorful-dark-hsl);--o-surface-1-visited-hsl: var(--o-surface-1-visited-colorful-dark-hsl);--o-surface-2-visited-hsl: var(--o-surface-2-visited-colorful-dark-hsl);--o-surface-3-visited-hsl: var(--o-surface-3-visited-colorful-dark-hsl);--o-surface-4-visited-hsl: var(--o-surface-4-visited-colorful-dark-hsl);--o-bw: var(--o-bw-colorful-dark);--o-bw-hsl: var(--o-bw-colorful-dark-hsl);--o-wb: var(--o-wb-colorful-dark);--o-wb-hsl: var(--o-wb-colorful-dark-hsl);--o-shadow-color: var(--o-shadow-color-colorful-dark);--o-shadow-strength: var(--o-shadow-strength-colorful-dark);--o-scrollbar-color: var(--o-scrollbar-color-colorful-dark);';
			$user_themes_css    .= '@media (prefers-color-scheme: dark) {[color-scheme="colorful"],[color-scheme="colorful-light"] .o-invert {' . $colorful_dark_theme . '}}[color-scheme="colorful-dark"],[color-scheme="colorful-light"] .o-invert {' . $colorful_dark_theme . '}';
		}

		return $user_themes_css;
	}

	/**
	 * Removes Gutenberg default styles and SVG presets
	 *
	 * @return void
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public function dequeue_gutenberg_global_styles() {
		// Remove WP CSS variables.
		remove_action( 'wp_enqueue_scripts', 'wp_enqueue_global_styles' );
		remove_action( 'wp_footer', 'wp_enqueue_global_styles', 1 );

		// Remove WP SVG definitions.
		remove_action( 'wp_body_open', 'wp_global_styles_render_svg_filters' );
	}

	/**
	 * Get useful Oxygen global settings
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string
	 */
	public function get_oxy_settings() {
		$custom_css = '';
		$defaults   = Defaults::get_instance();

		$current_settings = get_option( 'ct_global_settings', $defaults->get_original_global_settings_for_oxygen() );
		if ( $current_settings ) {
			$rem_to_px_ratio       = 16;
			$page_width            = $current_settings['max-width'];
			$max_viewport          = $page_width / $rem_to_px_ratio;
			$min_viewport          = $current_settings['breakpoints']['phone-portrait'] / $rem_to_px_ratio;
			$section_left_padding  = $current_settings['sections']['container-padding-left'];
			$section_right_padding = $current_settings['sections']['container-padding-right'];
			$h1                    = $current_settings['headings']['H1']['font-size'];
			$h1_unit               = $current_settings['headings']['H1']['font-size-unit'];
			$h2                    = $current_settings['headings']['H2']['font-size'];
			$h2_unit               = $current_settings['headings']['H2']['font-size-unit'];
			$h3                    = $current_settings['headings']['H3']['font-size'];
			$h3_unit               = $current_settings['headings']['H3']['font-size-unit'];
			$h4                    = $current_settings['headings']['H4']['font-size'];
			$h4_unit               = $current_settings['headings']['H4']['font-size-unit'];
			$h5                    = $current_settings['headings']['H5']['font-size'];
			$h5_unit               = $current_settings['headings']['H5']['font-size-unit'];
			$h6                    = $current_settings['headings']['H6']['font-size'];
			$h6_unit               = $current_settings['headings']['H6']['font-size-unit'];

			$custom_css = ":where(html){--o-oxy-page-width: {$page_width}px;--o-max_viewport: {$max_viewport};--o-min_viewport: {$min_viewport};--o-oxy-section-left-padding: {$section_left_padding};--o-oxy-section-right-padding: {$section_right_padding};--o-oxy-h1: {$h1};--o-oxy-h2: {$h2};--o-oxy-h3: {$h3};--o-oxy-h4: {$h4};--o-oxy-h5: {$h5};--o-oxy-h6: {$h6};}:where([class*=o-sub]){--o-oxy-h1: {$h1};--o-oxy-h2: {$h2};--o-oxy-h3: {$h3};--o-oxy-h4: {$h4};--o-oxy-h5: {$h5};--o-oxy-h6: {$h6};}";
		}
		return $custom_css;
	}

	/**
	 * Get useful Bricks settings
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string
	 */
	public function get_brx_settings() {
		$custom_css                               = '';
		$defaults                                 = Defaults::get_instance();
		$default_settings                         = array();
		$default_settings['oxyprops']['settings'] = $defaults->get_oxyprops_global_settings_for_bricks();
		$current_settings                         = get_option( 'bricks_theme_styles', $defaults->get_oxyprops_global_settings_for_bricks() );
		if ( $current_settings && isset( $current_settings['oxyprops'] ) && isset( $current_settings['oxyprops']['settings'] ) ) {
			$current_settings      = $current_settings['oxyprops']['settings'];
			$rem_to_px_ratio       = 16;
			$page_width            = isset( $current_settings['container']['widthMax'] ) ? preg_replace( '/[^0-9]/', '', $current_settings['container']['widthMax'] ) : 1360;
			$max_viewport          = $page_width / $rem_to_px_ratio;
			$min_viewport          = 478 / $rem_to_px_ratio;
			$section_left_padding  = isset( $current_settings['section']['padding']['left'] ) ? $current_settings['section']['padding']['left'] : '0';
			$section_right_padding = isset( $current_settings['section']['padding']['right'] ) ? $current_settings['section']['padding']['right'] : '0';
			$h1                    = isset( $current_settings['typography']['typographyHeadingH1']['font-size'] ) ? $current_settings['typography']['typographyHeadingH1']['font-size'] : ( isset( $current_settings['typography']['typographyHeadings']['font-size'] ) ? $current_settings['typography']['typographyHeadings']['font-size'] : 'var(--o-fluid-font)' );
			$h2                    = isset( $current_settings['typography']['typographyHeadingH2']['font-size'] ) ? $current_settings['typography']['typographyHeadingH2']['font-size'] : ( isset( $current_settings['typography']['typographyHeadings']['font-size'] ) ? $current_settings['typography']['typographyHeadings']['font-size'] : 'var(--o-fluid-font)' );
			$h3                    = isset( $current_settings['typography']['typographyHeadingH3']['font-size'] ) ? $current_settings['typography']['typographyHeadingH3']['font-size'] : ( isset( $current_settings['typography']['typographyHeadings']['font-size'] ) ? $current_settings['typography']['typographyHeadings']['font-size'] : 'var(--o-fluid-font)' );
			$h4                    = isset( $current_settings['typography']['typographyHeadingH4']['font-size'] ) ? $current_settings['typography']['typographyHeadingH4']['font-size'] : ( isset( $current_settings['typography']['typographyHeadings']['font-size'] ) ? $current_settings['typography']['typographyHeadings']['font-size'] : 'var(--o-fluid-font)' );
			$h5                    = isset( $current_settings['typography']['typographyHeadingH5']['font-size'] ) ? $current_settings['typography']['typographyHeadingH5']['font-size'] : ( isset( $current_settings['typography']['typographyHeadings']['font-size'] ) ? $current_settings['typography']['typographyHeadings']['font-size'] : 'var(--o-fluid-font)' );
			$h6                    = isset( $current_settings['typography']['typographyHeadingH6']['font-size'] ) ? $current_settings['typography']['typographyHeadingH6']['font-size'] : ( isset( $current_settings['typography']['typographyHeadings']['font-size'] ) ? $current_settings['typography']['typographyHeadings']['font-size'] : 'var(--o-fluid-font)' );
			$unit                  = str_contains( $page_width, 'px' ) ? '' : 'px';
			$custom_css            = ":where(html){--o-oxy-page-width: {$page_width}{$unit};--o-max_viewport: {$max_viewport};--o-min_viewport: {$min_viewport};--o-oxy-section-left-padding: {$section_left_padding};--o-oxy-section-right-padding: {$section_right_padding};--o-oxy-h1: {$h1};--o-oxy-h2: {$h2};--o-oxy-h3: {$h3};--o-oxy-h4: {$h4};--o-oxy-h5: {$h5};--o-oxy-h6: {$h6};}:where([class*=o-sub]){--o-oxy-h1: {$h1};--o-oxy-h2: {$h2};--o-oxy-h3: {$h3};--o-oxy-h4: {$h4};--o-oxy-h5: {$h5};--o-oxy-h6: {$h6};}";
		}
		return $custom_css;
	}

	/**
	 * User typography choices
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string
	 */
	public function get_user_typography_choices() {
		$defaults       = Defaults::get_instance();
		$typography     = isset( $this->oxyprops_current_settings['settings']['typography'] ) ? $this->oxyprops_current_settings['settings']['typography'] : $defaults->get_oxyprops_default_settings()['typography'];
		$css_to_return  = '@font-face{font-family: Atkinson Hyperlegible;src: url("/wp-content/plugins/oxyprops/public/fonts/Atkinson-Hyperlegible-Regular-102a.woff2") format("woff2"),url("/wp-content/plugins/oxyprops/public/fonts/Atkinson-Hyperlegible-Regular-102.woff") format("woff");font-weight: 400;}';
		$css_to_return .= '@font-face{font-family: Atkinson Hyperlegible;src: url("/wp-content/plugins/oxyprops/public/fonts/Atkinson-Hyperlegible-Bold-102a.woff2") format("woff2"),url("/wp-content/plugins/oxyprops/public/fonts/Atkinson-Hyperlegible-Bold-102.woff") format("woff");font-weight: 700;}';
		$css_to_return .= '@font-face{font-family: Atkinson Hyperlegible;src: url("/wp-content/plugins/oxyprops/public/fonts/Atkinson-Hyperlegible-Italic-102a.woff2") format("woff2"),url("/wp-content/plugins/oxyprops/public/fonts/Atkinson-Hyperlegible-Italic-102.woff") format("woff");font-weight: 400;font-style: italic;}';
		$css_to_return .= '@font-face{font-family: Atkinson Hyperlegible;src: url("/wp-content/plugins/oxyprops/public/fonts/Atkinson-Hyperlegible-BoldItalic-102a.woff2") format("woff2"),url("/wp-content/plugins/oxyprops/public/fonts/Atkinson-Hyperlegible-BoldItalic-102.woff") format("woff");font-weight: 700;font-style: italic;}';
		return $css_to_return . ':where(html){--o-desktop-to-mobile-ratio: ' . $typography['mobile_ratio'] . ';--o-base-text-size: ' . $typography['base_size'] . ';--o-stop-text-size: ' . $typography['minimum_size'] . ';--o-text-scale-ratio: ' . $typography['scale_ratio'] . ';--o-text-subheading-ratio: ' . $typography['subheading_ratio'] . ';}';
	}

	/**
	 * Converts color Hex value to an HSL array
	 *
	 * @since 1.1.3
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @param string $hex Hex code for the color.
	 *
	 * @return array
	 */
	public static function hexToHsl( $hex ) {
		$hex = array( $hex[1] . $hex[2], $hex[3] . $hex[4], $hex[5] . $hex[6] );
		$rgb = array_map(
			function ( $part ) {
				return hexdec( $part ) / 255;
			},
			$hex
		);

		$max = max( $rgb );
		$min = min( $rgb );

		$l = ( $max + $min ) / 2;

		if ( $max === $min ) {
			$h = 0;
			$s = 0;
		} else {
			$diff = $max - $min;
			$s    = $l > 0.5 ? $diff / ( 2 - $max - $min ) : $diff / ( $max + $min );

			switch ( $max ) {
				case $rgb[0]:
					$h = ( $rgb[1] - $rgb[2] ) / $diff + ( $rgb[1] < $rgb[2] ? 6 : 0 );
					break;
				case $rgb[1]:
					$h = ( $rgb[2] - $rgb[0] ) / $diff + 2;
					break;
				case $rgb[2]:
					$h = ( $rgb[0] - $rgb[1] ) / $diff + 4;
					break;
			}

			$h /= 6;

			$h = 360 * $h;
		}

		$s *= 100;
		$l *= 100;

		return array(
			'h' => $h . 'deg',
			's' => $s . '%',
			'l' => $l . '%',
		);
	}

	/**
	 * Converts color HSL array to an hex string
	 *
	 * @since 1.1.3
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @param array $hsl Array containing HSL components.
	 *
	 * @return string
	 */
	public static function hslToHex( $hsl ) {
		$hue        = $hsl['h'];
		$saturation = $hsl['s'];
		$lightness  = $hsl['l'];

		$h = floatval( substr( $hue, 0, -3 ) ) / 360;
		$s = floatval( substr( $saturation, 0, -1 ) ) / 100;
		$l = floatval( substr( $lightness, 0, -1 ) ) / 100;

		if ( 0 === $s ) {
			$s = 0.000001;
		}

		$q = $l < 0.5 ? $l * ( 1 + $s ) : $l + $s - $l * $s;
		$p = 2 * $l - $q;

		$r = self::hue2rgb( $p, $q, $h + 1 / 3 );
		$g = self::hue2rgb( $p, $q, $h );
		$b = self::hue2rgb( $p, $q, $h - 1 / 3 );

		return self::rgb2hex( $r ) . self::rgb2hex( $g ) . self::rgb2hex( $b );
	}
	/**
	 * Gets rgb from hue
	 *
	 * @param [type] $p P.
	 * @param [type] $q Q.
	 * @param [type] $t T.
	 * @return int
	 */
	public static function hue2rgb( $p, $q, $t ) {
		if ( $t < 0 ) {
			++$t;
		}
		if ( $t > 1 ) {
			--$t;
		}
		if ( $t < 1 / 6 ) {
			return $p + ( $q - $p ) * 6 * $t;
		}
		if ( $t < 1 / 2 ) {
			return $q;
		}
		if ( $t < 2 / 3 ) {
			return $p + ( $q - $p ) * ( 2 / 3 - $t ) * 6;
		}
		return $p;
	}

	/**
	 * Get Hex from rgb
	 *
	 * @param [type] $rgb RGB.
	 * @return string
	 */
	public static function rgb2hex( $rgb ) {
		return str_pad( dechex( round( $rgb * 255 ) ), 2, '0', STR_PAD_LEFT );
	}

	/**
	 * Register Bricks custom elements
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @return void
	 */
	public function register_bricks_elements() {
		if ( ! class_exists( '\Bricks\Elements' ) ) {
			return;
		}
		$defaults            = Defaults::get_instance();
		$elements            = $defaults->get_bricks_elements();
		$bricks_elements_dir = $this->oxyprops_path . 'includes/elements/bricks/classes/';

		foreach ( $elements as $key => $element ) {

			$file       = $bricks_elements_dir . $element['file_name'] . '.php';
			$name       = isset( $element['element_name'] ) ? $element['element_name'] : str_replace( '-', '', $element['file_name'] );
			$class_name = str_replace( '-', '_', $element['file_name'] );
			$class_name = ucwords( $class_name, '_' );

			// Register all elements in builder & frontend.
			\Bricks\Elements::register_element( $file, $name, $class_name );

			/* include styles in builder */
			if ( bricks_is_builder_iframe() ) {

				if ( ! isset( $element['stylesheet'] ) ) {
					$element['stylesheet'] = true;
				}

				if ( true === $element['stylesheet'] ) {
					$stylesheet = $name;
				} elseif ( false !== $element['stylesheet'] ) {
					$stylesheet = $element['stylesheet'];
				}

				if ( ! ! $element['stylesheet'] ) {
					wp_enqueue_style( $element['file_name'], $this->oxyprops_url . 'includes/elements/bricks/assets/css/' . $stylesheet . '.css', array(), $this->oxyprops_version );
				}
			}
		}
	}

	/**
	 * Register Bricks translatable strings
	 *
	 * @since 1.4.0
	 * @author Cédric Bontems <contact@oxyprops.com>
	 *
	 * @param array $i18n Strings.
	 *
	 * @return array
	 */
	public function register_bricks_strings( $i18n ) {
		// For element category 'custom'.
		$i18n['bricksprops'] = esc_html__( 'BricksProps', 'bricksprops' );
		return $i18n;
	}
}
